<?php

namespace App\Libraries;

use App\Models\TicketModel;
use App\Models\EventModel;
use App\Models\RegistryModel;
use App\Models\CategoryModel;
use App\Models\PhotoModel;
use App\Models\HistoryModel;

use stdClass;
use CodeIgniter\CLI\CLI;

class LibSincServer
{

    protected $baseUri;
    protected $urlPath;
    protected $client; // Curl Client
    protected $statusCode; //Curl response Header Code
    protected $headers; //Curl Headers
    protected $debug = false;

    private TicketModel $ticket;
    private EventModel $event;
    private CategoryModel $category;
    private RegistryModel $register;
    private PhotoModel $photo;
    private HistoryModel $history;

    public function __construct()
    {
        $this->ticket = new TicketModel();
        $this->event = new EventModel();
        $this->category = new CategoryModel();
        $this->register = new RegistryModel();
        $this->photo = new PhotoModel();
        $this->history = new HistoryModel();

        $this->urlPath = 'api/';
        $this->setUrl(URL_SYNC);
    }

    public function setUrl($baseUri)
    {
        $this->baseUri = $baseUri;
        $this->client = \Config\Services::curlrequest([
            'baseURI' => $this->baseUri,
            'timeout' => 0
        ]);
        return $this;
    }

    public function hasSync($data)
    {

        try {
            $request = $this->httpGET('sync/ticket', [
                'sort' => $data->sort,
                'order' => $data->order,
                'rows' => $data->rows,
                'page' => $data->page,
                'updated_at' => $data->updated_at,
            ]);
        } catch (\Exception $e) {
            CLI::write("| ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
            $request = false;
        }
        return $request;
    }

    public function updateSync($file)
    {

        $spl = (object) [
            'index' => 0,
            'active' => 0,
            'inactive' => 0,
            'proccess' => 0,
            'updated_at' => 0,
        ];
        $lines = count($file->data->tickets);
        foreach ($file->data->tickets as $tickets) {

            if (!empty($tickets)) {
                $spl->active++;

                CLI::clearScreen();
                CLI::showProgress($spl->index, $lines);
                CLI::write('|=====================================================================|');
                CLI::write("| UPDATE: tickets {$tickets->ticket_number} | {$tickets->name}      |", 'dark_blue', 'yellow');
                CLI::write('|=====================================================================|');

                CLI::write('|=====================================================================|');

                if (strcmp((string) $spl->updated_at, (string) $tickets->updated_at) < 0) {
                    $spl->updated_at = $tickets->updated_at;
                }
                if ($this->updateTicket($tickets)) {
                    $spl->proccess++;
                }
            } else {
                $spl->inactive++;
            }
            $spl->index++;
        }

        return $spl;
    }

    private function updateTicket($fields)
    {
        $sts = false;
        $ticket = $this->ticket->where([
            'cartao' => $fields->ticket_number,
            'id_evento' => $fields->event_id
        ])->first();

        if (
            $this->validateCategory($fields) &&
            $this->validateEvent($fields)
        ) {
            if (!empty($ticket)) {

                if (
                    !empty($ticket->send_on)
                    && strcmp((string) $fields->updated_at, (string) $ticket->updated_at) > 0
                    && strcmp((string) $fields->updated_at, (string) $ticket->send_on) > 0
                ) {
                    $data = [
                        'nome' => $fields->name,
                        'campoextra1' => $fields->extrafield_1,
                        'campoextra2' => $fields->extrafield_2,
                        'updated_at' => $fields->updated_at,
                        'nome' => $fields->name,
                        'email' => $fields->email,
                        'situacao' => $fields->status,
                        'id_categoria' => $fields->category_id,
                        'send_on' => $fields->updated_at,
                    ];

                    try {
                        $historyData = json_encode($ticket);
                        $insertHistory = [
                            'tabela' => 'ticket',
                            'tabela_id' => $ticket->id,
                            'data_json' => ($historyData),
                            'created_at' => date('Y-m-d H:i:s'),
                        ];
                        $this->history->insert($insertHistory);
                        $this->ticket->update($ticket->id, $data);
                        $sts = true;
                    } catch (\Exception $e) {
                        CLI::write("| ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
                        $sts = false;
                    }
                }
            } else {

                $data = [
                    'nome' => $fields->name,
                    'campoextra1' => $fields->extrafield_1,
                    'campoextra2' => $fields->extrafield_2,
                    'updated_at' => $fields->updated_at,
                    'send_on' => $fields->updated_at,
                    'nome' => $fields->name,
                    'email' => $fields->email,
                    'situacao' => $fields->status,
                    'id_categoria' => $fields->category_id,
                    'id_evento' => $fields->event_id,
                    'cartao' => $fields->ticket_number,
                ];

                try {
                    $this->ticket->insert($data);
                    $sts = true;
                } catch (\Exception $e) {
                    $cart = $data['cartao'];
                    CLI::write("| TICKET ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
                    CLI::write("| {$cart} e {$fields->ticket_number}  ", 'dark_blue', 'red');
                    sleep(5);
                    $sts = false;
                }
            }
        } else {
            $sts = false;
        }

        return $sts;
    }

    public function updateRegisterSync($file)
    {

        $spl = (object) [
            'index' => 0,
            'active' => 0,
            'inactive' => 0,
            'proccess' => 0,
            'data_acesso' => 0,
        ];
        $lines = count($file->data->register);
        foreach ($file->data->register as $register) {

            if (!empty($register)) {
                $spl->active++;

                CLI::clearScreen();
                CLI::showProgress($spl->index, $lines);
                CLI::write('|=====================================================================|');
                CLI::write("| UPDATE: Register {$register->cartao} | {$register->data_acesso}      |", 'dark_blue', 'yellow');
                CLI::write('|=====================================================================|');

                CLI::write('|=====================================================================|');

                if (strcmp((string) $spl->data_acesso, (string) $register->data_acesso) < 0) {
                    $spl->data_acesso = $register->data_acesso;
                }
                if ($this->updateRegister($register)) {
                    $spl->proccess++;
                }
            } else {
                $spl->inactive++;
            }
            $spl->index++;
        }



        return $spl;
    }


    public function updateRegister($fields)
    {

        $sts = false;
        $register = $this->register->where([
            'cartao' => $fields->cartao,
            'data_acesso' => $fields->data_acesso
        ])->first();

        $ticket = $this->ticket->where([
            'cartao' => $fields->cartao
        ])->first();

        if (empty($register) && !empty($ticket)) {

            $data = [
                'id_pessoa' => $ticket->id,
                'id_evento' => $ticket->id_evento,
                'cartao' => $fields->cartao,
                'data_acesso' => $fields->data_acesso,
                'hora_acesso' => $fields->hora_acesso,
                'acao' => $fields->acao,
                'terminal' => $fields->terminal,
                'giro' => $fields->giro,
                'send_on' => $fields->data_acesso
            ];

            try {
                $this->register->insert($data);
                $sts = true;
            } catch (\Exception $e) {
                CLI::write("| REGISTER ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
                CLI::write("| {$fields->cartao} - {$fields->data_acesso}  ", 'dark_blue', 'red');
                sleep(5);
                $sts = false;
            }
        }

        return $sts;
    }

    private function validateCategory($fields)
    {
        $category = $this->category->find($fields->category_id);

        $sts = false;
        if (empty($category)) {

            $data = [
                'id' => $fields->category_id,
                'categoria' => $fields->category,
                'tipo' => $fields->category_type,
                'idr_multiplo' => $fields->category_idr_multiplo,
                'fluxo' => $fields->category_idr_multiplo
            ];

            try {
                $this->category->insert($data);
                $sts = true;
            } catch (\Exception $e) {
                CLI::write("| CATEGORY ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
                CLI::write("| CATEGORY {$fields->category_id}  ", 'dark_blue', 'red');
                $sts = false;
                sleep(5);
            }
        } else {
            $sts = true;
        }

        return $sts;
    }
    private function validateEvent($fields)
    {
        $event = $this->event->find($fields->event_id);

        $sts = false;
        if (empty($event)) {
            $data = [
                'id' => $fields->event_id,
                'evento' => $fields->event,
                'local' => $fields->event_local,
                'data_inicio' => $fields->event_start,
                'data_fim' => $fields->event_end,
                'campoextra1' => $fields->event_extra1,
                'campoextra2' => $fields->event_extra2
            ];

            try {
                $this->event->insert($data);
                $sts = true;
            } catch (\Exception $e) {
                CLI::write("| EVENT ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
                CLI::write("| EVENT {$fields->event_id}  ", 'dark_blue', 'red');
                $sts = false;
                sleep(5);
            }
        } else {
            $sts = true;
        }

        return $sts;
    }

    public function hasRegister($data)
    {

        try {
            $sendRequest = [
                'sort' => $data->sort,
                'order' => $data->order,
                'rows' => $data->rows,
                'page' => $data->page,
                // 'ticket_number'=> '11844190',  // remover
            ];

            if (!empty($data->data_maior)) {
                $sendRequest['data_maior'] = $data->data_maior;
            }

            $request = $this->httpGET('sync/register', $sendRequest);
        } catch (\Exception $e) {
            CLI::write("| ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
            $request = false;
        }
        return $request;
    }

    public function hasPhoto($data)
    {
        try {
            $sendRequest = [
                'sort' => 'id',
                'order' => 'asc',
                'rows' => 10,
                'page' => $data->page,
            ];

            if (!empty($data->data_maior)) {
                $sendRequest['data_maior'] = $data->created_at;
            }

            $request = $this->httpGET('sync/photo', $sendRequest);
        } catch (\Exception $e) {
            CLI::write("| ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
            $request = false;
        }
        return $request;
    }

    public function updatePhotoSync($file)
    {
        $spl = (object) [
            'index' => 0,
            'active' => 0,
            'inactive' => 0,
            'proccess' => 0,
            'created_at' => null,
        ];
        $lines = count($file->data->photo);
        foreach ($file->data->photo as $photo) {
            if (!empty($photo)) {

                $spl->active++;

                CLI::clearScreen();
                CLI::showProgress($spl->index, $lines);
                $dateNow = date('d/m/Y H:i:s');
                CLI::write('|=====================================================================|');
                CLI::write("| UPDATE: Photo {$photo->id} | {$photo->ticketNumber} -  {$photo->created_at}  em: {$dateNow}    |", 'dark_blue', 'yellow');
                CLI::write('|=====================================================================|');

                CLI::write('|=====================================================================|');

                if (strcmp((string) $spl->created_at, (string) $photo->created_at) < 0) {
                    $spl->created_at = $photo->created_at;
                }
                if ($this->updatePhoto($photo)) {
                    $spl->proccess++;
                }
            } else {
                $spl->inactive++;
            }
            $spl->index++;
        }
        return $spl;
    }

    public function updatePhoto($fields)
    {
        $sts = false;

        $ticket = $this->ticket->where([
            'cartao' => $fields->ticketNumber
        ])->first();
   
        if (!empty($ticket)) {

            $photo = $this->photo->where([
                'id_pessoas' => $ticket->id
            ])
                ->first();

            if (!empty($photo)) {
                if (strcmp((string) $photo->criado_em, (string) $fields->created_at) < 0 && !empty($photo->send_on)) {
                    $data = [
                        'foto' => base64_decode($fields->image),
                        'send_on' => $fields->created_at,
                    ];
             
                    try {
                        $this->photo->update($photo->id, $data);
                        $sts = true;
                    } catch (\Exception $e) {
                        CLI::write("| PHOTO UPDATE ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
                        CLI::write("| {$fields->ticketNumber} - {$fields->created_at}  ", 'dark_blue', 'red');
                        sleep(5);
                        $sts = false;
                    }
                }
            } else {
                $data = [
                    'id_pessoas' => $ticket->id,
                    'foto' => base64_decode($fields->image),
                    'send_on' => $fields->created_at,
                    'criado_em' => $fields->created_at,
                ];

                try {
                    $this->photo->insert($data);
                    $sts = true;
                } catch (\Exception $e) {
                    CLI::write("| PHOTO INSERT ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
                    CLI::write("| {$fields->ticketNumber} - {$fields->created_at}  ", 'dark_blue', 'red');
                    sleep(5);
                    $sts = false;
                }
            }
        }

        return $sts;
    }


    public function getUnsendedRegister($limit)
    {
        return $this->register
            ->select(
                'id,
                cartao,
                data_acesso,
                acao,
                terminal,
                giro,
                now() as send_on'
            )
            ->where([
                'send_on' => null
            ])->findAll($limit);
    }

    public function getUnsendedTicket($limit)
    {
        return $this->ticket
            ->select(
                'id,
                id_user,
                id_categoria,
                cartao,
                nome,
                email,
                rg,
                cpf,
                situacao,
                campoextra1,
                campoextra2,
                mestre,
                now() as send_on,
                updated_at'
            )
            ->where([
                'send_on' => null
            ])->findAll($limit);
    }

    public function getUnsendedPhoto($limit)
    {
        return $this->photo
            ->select(
                'fotos.id,
                cartoes.cartao,
                fotos.id_pessoas,
                fotos.foto,
                now() as send_on,
                fotos.criado_em'
            )
            ->where([
                'fotos.send_on' => null
            ])->join('cartoes', 'cartoes.id = fotos.id_pessoas', 'left')
            ->findAll($limit);
    }

    public function sendRegister($list)
    {

        try {
            $request = $this->httpPOST('sync/register', $list);
        } catch (\Exception $e) {
            CLI::write("| ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
            $request = false;
        }

        if ($request) {

            if ($request && property_exists($request, 'status') && $request->status && property_exists($request, 'data') && property_exists($request->data, 'valid')) {

                foreach ($request->data->valid as $register) {

                    $data = [
                        'send_on' => $register->send_on
                    ];

                    try {
                        $this->register->update($register->id, $data);
                        $sts = true;
                    } catch (\Exception $e) {
                        CLI::write("| REGISTER ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
                        CLI::write("| {$register->cartao} - {$register->data_acesso}  ", 'dark_blue', 'red');
                        sleep(1);
                    }
                }
            }
        }
    }

    public function sendTickets($list)
    {

        try {
            $request = $this->httpPOST('sync/ticket', $list);
        } catch (\Exception $e) {
            CLI::write("| ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
            $request = false;
        }

        if ($request) {

            if ($request && property_exists($request, 'status') && $request->status && property_exists($request, 'data')) {

                foreach ($request->data->new as $external) {

                    $data = [
                        'send_on' => $external->send_on
                    ];

                    try {
                        $this->ticket->update($external->id, $data);
                        $sts = true;
                    } catch (\Exception $e) {
                        CLI::write("| TICKET ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
                        CLI::write("| {$external->cartao} - {$external->send_on}  ", 'dark_blue', 'red');
                        sleep(1);
                    }
                }

                foreach ($request->data->update as $external) {

                    $data = [
                        'send_on' => $external->send_on
                    ];

                    try {
                        $this->ticket->update($external->id, $data);
                        $sts = true;
                    } catch (\Exception $e) {
                        CLI::write("| TICKET ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
                        CLI::write("| {$external->cartao} - {$external->send_on}  ", 'dark_blue', 'red');
                        sleep(1);
                    }
                }
            }
        }
    }

    public function sendPhotos($list)
    {

        $data = [];
        foreach ($list as $photo) {
            $reg = [
                'id' => $photo->id,
                'cartao' => $photo->cartao,
                'id_pessoas' => $photo->id_pessoas,
                'foto' => null,
                'send_on' => $photo->send_on,
                'criado_em' => $photo->criado_em
            ];

            if (!empty($photo->foto)) {
                $reg['foto'] = base64_encode($photo->foto);
            } else {
                continue;
            }

            $data[] = $reg;
        }

        try {
            $request = $this->httpPOST('sync/photo', $data);
        } catch (\Exception $e) {
            CLI::write("| ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
            $request = false;
        }

        

        if ($request) {

            if ($request && property_exists($request, 'status') && $request->status && property_exists($request, 'data')) {
           
                foreach ($request->data->valid as $external) {

                    $data = [
                        'send_on' => $external->send_on
                    ];

                    try {
                        $this->photo->update($external->id, $data);
                        $sts = true;
                    } catch (\Exception $e) {
                        CLI::write("| PHOTO ERROR:  - {$e->getMessage()}   ", 'dark_blue', 'red');
                        CLI::write("| {$external->cartao} - {$external->send_on}  ", 'dark_blue', 'red');
                        sleep(1);
                    }
                }
            }
        }
    }

    /**
     * get Request Status Code (200, 401 ...)
     * @return integer
     */
    public function getStatusCode()
    {
        return (int) $this->statusCode;
    }

    /**
     * Set debug as active
     * @param type $is
     */
    public function setDebug($is = true)
    {
        $this->debug = (bool) $is;
        return $this;
    }

    /**
     * Generator Header
     */
    private function apiAuth()
    {
        $this->clearHeader();
        $this->setHeader('Cache-Control', 'no-cache');
        $this->setHeader('Connection', 'keep-alive');
        $this->setHeader('User-Agent', 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/97.0.4692.99 Safari/537.36');
    }

    /**
     * Cleare header
     */
    public function clearHeader()
    {
        $this->headers = array();
    }

    /**
     * Set Header
     * @param string $key Content-type
     * @param string $value Application/json
     */
    public function setHeader($key, $value)
    {
        $this->headers[$key] = $value;
    }

    /**
     * Request GET
     * @param string $method
     * @param mixed $ardatarayParam
     * @return content
     */
    public function httpGET($method, $ardatarayParam = [])
    {

        $query = '';
        if (!empty($ardatarayParam)) {
            if (is_array($ardatarayParam)) {
                $query = '?' . http_build_query($ardatarayParam);
            } else {
                $query = '?' . $ardatarayParam;
            }
        }

        $this->apiAuth();
        $options = ['headers' => $this->headers];
        $options['baseURI'] = $this->baseUri;
        $options['http_errors'] = false;
        $options['verify'] = false;
        $options['connect_timeout'] = 0;
        if ($this->debug) {
            $options['debug'] = FCPATH . '../writable/curl_log.txt';
        }

        $methosPath = $this->urlPath . $method;
        $options['baseURI'] = $this->baseUri;
        $response = $this->client->request('GET', $methosPath . $query, $options);
        $this->statusCode = $response->getStatusCode();
        return $this->content($response);
    }

    /**
     * Request POST
     * @param string $method
     * @param mixed $data
     * @param array $header
     * @return content
     */
    public function httpPOST($method, $data = null, array $header = null)
    {

        $this->apiAuth();

        if (!empty($header)) {
            foreach ($header as $key => $value) {
                $this->headers[$key] = $value;
            }
        }
        $this->headers['Connection'] = 'keep-alive';
        $this->headers['Content-Type'] = 'application/json';
        foreach ($this->headers as $key => $value) {
            $this->client->setHeader($key, $value);
        }
        $ijdata = json_encode($data, JSON_PRETTY_PRINT);
        //var_dump($ijdata);exit();
        $this->client->setBody($ijdata);
        $options = [];
        $options['baseURI'] = $this->baseUri;
        $options['http_errors'] = false;
        $options['verify'] = false;
        $options['connect_timeout'] = 0;
        if ($this->debug) {
            $options['debug'] = FCPATH . '../writable/curl_log.txt';
        }

        $methosPath = $this->urlPath . $method;
        $response = $this->client->request('POST', $methosPath, $options);

        $this->statusCode = $response->getStatusCode();

        return $this->content($response);
    }

    /**
     * get response content
     * @param response $request
     * @return mixed
     */
    public function content($request)
    {
        $body = '';
        $Encoding = trim($request->getHeaderLine('Content-Encoding'));
        if ($Encoding === 'gzip') {
            $body = gzdecode($request->getBody());
        } else {
            $body = $request->getBody();
        }

        $type = trim($request->getHeaderLine('Content-Type'));
        if (strpos($type, 'json')) {
            $body = json_decode($body);
        }

        return $body;
    }
}
