<?php

namespace App\Libraries;

use CodeIgniter\Model;

class LibPaginate
{

    private $model;
    private $request;
    private $rows;
    private $page;
    private $sort;
    private $label;
    private $localFilter;
    private $isJsDatabase;
    private $columns;

    public function __construct(Model $model, \CodeIgniter\HTTP\RequestInterface $request, string $label = 'data', array $columns = [])
    {
        $this->model = $model;
        $this->request = $request;
        $this->label = $label;
        $this->localFilter = [];
        $this->rows = 50;
        $this->page = 1;
        $this->columns = $columns;
        $this->isJsDatabase = !empty($this->columns);
        $this->load();
    }

    private function load()
    {
        if($this->isJsDatabase) {
            $start = (int) $this->request->getGet('start');
            $qtd = (int) $this->request->getGet('length');
            $busca = $this->request->getGet('search');
            $ordem = $this->request->getGet('order');
            $sort = $this->columns[$ordem[0]['column']];

            $order = $ordem[0]['dir'];
            $page = 1;

            if ($start > 0 && $start >= $qtd) {
                $page = intval($start / $qtd) + 1;
            }

            $this->setSort($sort);
            $this->setOrder($order);
            $this->setPage($page);
            $this->setRows($qtd);
        } else {
            $this->setRows($this->request->getGet('rows'));
            $this->setPage($this->request->getGet('page'));
            $this->setSort($this->request->getGet('sort'));
            $this->setOrder($this->request->getGet('order'));
        }

    }


    public function setRows($rows)
    {
        if ($rows) {
            $this->rows = intval($rows);
        }
    }

    public function setPage($page)
    {
        if ($page) {
            $this->page = intval($page);
        }
    }

    public function useFields(array $fields)
    {
        $this->model->select(implode(",", $fields));
    }

    public function setSort($field)
    {
        if($this->isJsDatabase) {
            $this->sort = $field;
        } else {
            if (!empty($field) && in_array($field, $this->model->allowedFields)) {
                $this->sort = $field;
            } else if (in_array('id', $this->model->allowedFields)) {
                $this->sort = 'id';
            }
        }
    }

    public function setOrder($order, $field = null)
    {
        if (!empty($field)) $this->setSort($field);

        if (!empty($order) && !empty($this->sort)) {
            $order = trim(strtolower($order??''));
            if ($order == "asc") {
                $this->model->orderBy($this->sort, $order);
            }
            if ($order == "desc") {
                $this->model->orderBy($this->sort, $order);
            }
        }
    }

    public function addFilter($queryField, $requestField, $type = '=')
    {
        if ($this->request->getGetPost($requestField)) {
            $this->localFilter[] = (object)[
                'field' => $queryField,
                'requestField' => $requestField,
                'value' => $this->request->getGetPost($requestField),
                'type' => $type
            ];
        }
    }

    private function filter()
    {
        foreach ($this->localFilter as $filter) {
            if($this->isJsDatabase && $filter->value == 'null') {
                continue;
            }
            if ($filter->type == '=') {
                $this->model->where($filter->field, $filter->value);
            }
            if ($filter->type == 'like') {
                $this->model->like($filter->field, $filter->value);
            }
            if ($filter->type == '>') {
                $this->model->where($filter->field.' >', $filter->value);
            }
            if ($filter->type == '<') {
                $this->model->where($filter->field.' <', $filter->value);
            }
            if ($filter->type == '>=') {
                $this->model->where($filter->field.' >=', $filter->value);
            }
            if ($filter->type == 'maior') {
                $this->model->where($filter->field.' >=', $filter->value);
            }
            if ($filter->type == '<=') {
                $this->model->where($filter->field.' <=', $filter->value);
            }
            if ($filter->type == 'null') {
                $this->model->where($filter->field, null);
            }
            if ($filter->type == 'notnull') {
                $this->model->where("{$filter->field} is not null");
            }
        }
    }
    private function pagination()
    {
        return (object) [
            'rows' => $this->model->pager->getTotal('default'),
            'pages' => $this->model->pager->getPageCount('default'),
            'current' => $this->model->pager->getCurrentPage('default'),
        ];
    }

    private function result()
    {
        return $this->model->paginate($this->rows, 'default', $this->page, 1);
    }

    public function getFilter()
    {
        return array_map(function ($e) {
            return (object)[
                'field' => $e->requestField,
                'value' => $e->value,
            ];
        }, $this->localFilter);
    }

    public function get()
    {
        $this->filter();
        return [
            $this->label => $this->result(),
            'pagination' => $this->pagination(),
            'filter' => $this->getFilter(),
        ];
    }
    public function getData()
    {
        $this->filter();
        return $this->model->findAll();
    }
}
