<?php

namespace App\Libraries\Cloudid;

class CloudidHttp
{

    protected $version = '1.0.0';
    protected $token;
    protected $baseUri;
    protected $urlPath;
    protected $client; // Curl Client
    protected $statusCode; //Curl response Header Code
    protected $headers; //Curl Headers
    protected $debug = false;
    protected $logFile = false;

    public function __construct()
    {
        $this->urlPath = '/api/v2/';
        $this->setUrl('https://reentrada.ticketeria.tk');
        $this->logFile = FCPATH . '../writable/curl_log.txt';
    }

    /**
     * get library version
     * @return string
     */
    public function getVersion()
    {
        return $this->version;
    }

    /**
     * Set Api-key
     * @param string $apiKey
     */
    private function setUrl($baseUri)
    {
        $this->baseUri = $baseUri;
        $this->client = \Config\Services::curlrequest([
            'baseURI' => $this->baseUri,
            'timeout' => 0
        ]);
        return $this;
    }
   

    /**
     * get Request Status Code (200, 401 ...)
     * @return integer
     */
    public function getStatusCode()
    {
        return (int) $this->statusCode;
    }

    /**
     * Set debug as active
     * @param type $is
     */
    public function setDebug($is = true)
    {
        $this->debug = (bool) $is;
        return $this;
    }

    /**
     * Generator Header
     */
    private function apiAuth()
    {
        $this->clearHeader();
        $this->setHeader('Cache-Control', 'no-cache');
        $this->setHeader('Connection', 'keep-alive');
        $this->setHeader('Accept', '*/*');
        $this->setHeader('User-Agent', 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/97.0.4692.99 Safari/537.36');
    }

    /**
     * Cleare header
     */
    public function clearHeader()
    {
        $this->headers = array();
    }

    /**
     * Set Header
     * @param string $key Content-type
     * @param string $value Application/json
     */
    public function setHeader($key, $value)
    {
        $this->headers[$key] = $value;
    }

    /**
     * Request GET
     * @param string $method
     * @param mixed $ardatarayParam
     * @return content
     */
    public function httpGET($method, $ardatarayParam = [])
    {
        $query = '';
        if (!empty($ardatarayParam)) {
            if (is_array($ardatarayParam)) {
                $query = '?' . http_build_query($ardatarayParam);
            } else {
                $query = '?' . $ardatarayParam;
            }
        }

        $this->apiAuth();
        $options = ['headers' => $this->headers];
        $options['baseURI'] = $this->baseUri;
        $options['http_errors'] = false;
        $options['verify'] = false;
        $options['connect_timeout'] = 0;
        if ($this->debug) {
            $options['debug'] = $this->logFile;
        }

        $methosPath = $this->urlPath . $method;
        $options['baseURI'] = $this->baseUri;
        $response = $this->client->request('GET', $methosPath . $query, $options);
        $this->statusCode = $response->getStatusCode();
  
        return $this->content($response);
    }

    /**
     * Request DELETE
     * @param string $method
     * @param mixed $ardatarayParam
     * @return content
     */
    public function httpDELETE($method, $ardatarayParam = [])
    {
        $query = '';
        if (!empty($ardatarayParam)) {
            if (is_array($ardatarayParam)) {
                $query = '?' . http_build_query($ardatarayParam);
            } else {
                $query = '?' . $ardatarayParam;
            }
        }

        $this->apiAuth();
        $options = ['headers' => $this->headers];
        $options['http_errors'] = false;
        $options['baseURI'] = $this->baseUri;
        $options['verify'] = false;
        $options['connect_timeout'] = 0;
        if ($this->debug) {
            $options['debug'] = $this->logFile;
        }
        $methosPath = $this->urlPath . $method;

        $response = $this->client->request('DELETE', $methosPath . $query, $options);

        $this->statusCode = $response->getStatusCode();
        return $this->content($response);
    }

    /**
     * Request POST
     * @param string $method
     * @param mixed $data
     * @param array $header
     * @return content
     */
    public function httpPOST($method, $data = null, array $header = null)
    {

        $this->apiAuth();

        if (!empty($header)) {
            foreach ($header as $key => $value) {
                $this->headers[$key] = $value;
            }
        }
        $this->headers['Connection'] = 'keep-alive';
        $this->headers['Content-Type'] = 'application/json';
        foreach ($this->headers as $key => $value) {
            $this->client->setHeader($key, $value);
        }
        $ijdata = json_encode($data, JSON_PRETTY_PRINT);
        $this->client->setBody($ijdata);
        $options = [];
        $options['baseURI'] = $this->baseUri;
        $options['http_errors'] = false;
        $options['verify'] = false;
        $options['connect_timeout'] = 0;
        if ($this->debug) {
            $options['debug'] = $this->logFile;
        }

        $methosPath = $this->urlPath . $method;
        $response = $this->client->request('POST', $methosPath, $options);

        $this->statusCode = $response->getStatusCode();

        return $this->content($response);
    }

    /**
     * Request POST
     * @param string $method
     * @param mixed $data
     * @param array $header
     * @return content
     */
    public function httpPUT($method, $data = null, array $header = null)
    {

        $this->apiAuth();

        if (!empty($header)) {
            foreach ($header as $key => $value) {
                $this->headers[$key] = $value;
            }
        }
        $this->headers['Connection'] = 'keep-alive';
        $this->headers['Content-Type'] = 'application/json';
        foreach ($this->headers as $key => $value) {
            $this->client->setHeader($key, $value);
        }
        $ijdata = json_encode($data, JSON_PRETTY_PRINT);
        $this->client->setBody($ijdata);
        $options = [];
        $options['baseURI'] = $this->baseUri;
        $options['http_errors'] = false;
        $options['verify'] = false;
        $options['connect_timeout'] = 0;
        if ($this->debug) {
            $options['debug'] = $this->logFile;
        }

        $methosPath = $this->urlPath . $method;
        $response = $this->client->request('PUT', $methosPath, $options);

        $this->statusCode = $response->getStatusCode();

        return $this->content($response);
    }

    /**
     * get response content
     * @param response $request
     * @return mixed
     */
    public function content($request)
    {
        $body = '';
        $encoding = trim($request->getHeaderLine('Content-Encoding'));
        
        if ($encoding === 'gzip') {
            $body = gzdecode($request->getBody());
        } else {
            $body = $request->getBody();
        }
        
        $type = trim($request->getHeaderLine('Content-Type'));
        if (strpos($type, 'json')) {
            $body = json_decode($body);
        }
        return $body;
    }

}
