<?php

namespace App\Controllers\Event;

use App\Controllers\BaseController;
use App\Models\TicketModel;
use App\Models\EventModel;
use App\Models\CategoryModel;
use App\Models\PhotoModel;
use App\Models\RegistryModel;
use App\Entities\TicketEntity;
use App\Libraries\LibPaginate;

class Ticket extends BaseController {

    private TicketEntity $ticket;
    private EventModel $event;
    private CategoryModel $category;
    private PhotoModel $photo;
    private RegistryModel $registry;

    public function __construct()
    {
        $this->model = new TicketModel();
        $this->ticket = new TicketEntity();
        $this->event = new EventModel();
        $this->category = new CategoryModel();
        $this->photo = new PhotoModel();
        $this->registry = new RegistryModel();
    }

    public function initController($request, $response, $logger)
    {
        // Do Not Edit This Line
        parent::initController($request, $response, $logger);
        if (!$this->authUser->logged()) {
            return redirect()->route("auth/signin");
        }
    }

    public function index() {
        $this->addOnView('events', $this->event->findAll());
        $this->addOnView('categories', $this->category->findAll());
        return view('ticket/index', $this->getViewData());
    }

    public function add()
    {
        $this->addOnView('ticket', $this->ticket);
        $this->addOnView('events', $this->event->findAll());
        $this->addOnView('categories', $this->category->findAll());
        $this->addOnView('registry', []);
        return view('ticket/edit', $this->getViewData());
    }

    public function edit($id)
    {
        $url = previous_url() ?? base_url();
        $result = $this->model->find((int)$id);
        if ($result) {
            $this->ticket->fill((array)$result);
            $rsRegistry = $this->registry->getRegistry(intval($id));
            $this->addOnView('events', $this->event->findAll());
            $this->addOnView('categories', $this->category->findAll());
            $this->addOnView('ticket', $result);
            $this->addOnView('registry', $rsRegistry);
            $this->addOnView('acts', $this->registry->getAcoes());
            return view('ticket/edit', $this->getViewData());
        } else {
            $this->authUser->setFlashdata('error', 'Evento inválido');
            return redirect()->to($url);
        }
    }

    public function delete($id)
    {
        $url = previous_url() ?? base_url();
        $result = $this->model->find((int)$id);
        if ($result) {
            $this->model->delete($id);
            $this->authUser->setFlashdata('success', 'Registro removido com sucesso');
            return redirect()->to('/ticket');
        } else {
            $this->authUser->setFlashdata('error', 'Registro inválido');
            return redirect()->to($url);
        }
    }

    public function addsave()
    {
        $this->ticket->fill($this->request->getPost());
        $url = previous_url() ?? base_url();
        if (empty($this->ticket->situacao)) {
            $this->ticket->situacao = 0;
        }
        $update = $this->model->save($this->ticket->toArray());

        if (!empty($update)) {
            $this->authUser->setFlashdata('success', 'Evento criado com sucesso');
            return redirect()->to('/ticket');
        } else {
            $this->authUser->setFlashdata('error', $update->message);
            return redirect()->to($url);
        }
    }

    public function save($id)
    {
        $url = previous_url() ?? base_url();
        $this->ticket->fill($this->request->getPost());
        $this->ticket->id = $id;
        if (empty($this->ticket->situacao)) {
            $this->ticket->situacao = 0;
        }
        $result = $this->model->save($this->ticket->toArray());
        if ($result ) {
            $this->authUser->setFlashdata('success', 'Ticket atualizado com sucesso');
            return redirect()->to('/ticket');
        } else {
            $this->authUser->setFlashdata('error', 'ticket inválido');
            return redirect()->to($url);
        }
    }

    public function photo($id)
    {
        $result = $this->photo->where('id_pessoas', intval($id))->first();

        if ($result && $result->foto && strlen($result->foto) > 10) {
            $image = $result->foto;
        } else {
            $image = file_get_contents(WRITEPATH . 'uploads/person.jpg');
        }
        
        $this->response->setHeader('Cache-Control', 'no-cache')
        ->appendHeader('Content-Type', 'image/jpeg');
        echo $image;
    }
    public function getPhoto($id)
    {
        $result = $this->photo->where('id_pessoas', intval($id))->first();

        $output = (object)[
            'status'=>false,
            'id'=>null,
            'image'=>null,
            'message'=>null,
        ];
        if ($result && $result->foto && strlen($result->foto) > 10) {
            $output->status = true;
            $output->id = $result->id;
            $output->image = 'data:image/jpeg;base64,' . base64_encode($result->foto);
        } else {
            $output->message = 'foto inválida';
            $output->image = 'data:image/jpeg;base64,' . base64_encode(file_get_contents(WRITEPATH . 'uploads/person.jpg'));
        }
        return $this->response->setJSON($output);
    }
    
    public function getRegistry($id)
    {
        $result = $this->registry->where('id_pessoa', intval($id))->orderBy('id', 'desc')->limit(30)->findAll();
        $output = (object)[
            'status'=>false,
            'id'=>null,
            'image'=>null,
            'message'=>null,
        ];
        if ($result) {
            $output->status = true;
            $output->registry = $result;
        } else {
            $output->message = 'sem registro';
        }
        return $this->response->setJSON($output);
    }

    public function data()
    {
        $data = [];
        $orders = ['id', 'cartao', 'cartoes.id_evento', 'cartoes.situacao', 'cartoes.create_time', '', ''];
        $busca = $this->request->getGet('search');

        $model = $this->model->select("
                        cartoes.id,
                        cartoes.id_evento,
                        cartoes.id_categoria,
                        cartoes.cartao,
                        cartoes.nome,
                        cartoes.situacao,
                        cartoes.mestre,
                        cartoes.create_time,
                        cartoes.updated_at,
                        cartoes.send_on,
                        eventos.evento,
                        categorias.categoria
                ")
                ->join('eventos', 'eventos.id = cartoes.id_evento', 'inner')
                ->join('categorias', 'categorias.id = cartoes.id_categoria', 'inner')
                ->groupBy('cartoes.id');
        if ($busca && $busca['value']) {
            $qry = trim((string) $busca['value']);
            $model->where(" concat(cartoes.cartao, ' ', cartoes.id, ' ', ifnull(cartoes.nome,''), ' ', ifnull(cartoes.rg,''), ' ', ifnull(cartoes.cpf,''), ' ', ifnull(cartoes.email,'')) like '%$qry%'");
        }

        $pagination = new LibPaginate($model, $this->request, 'data', $orders);

        $pagination->addFilter('cartoes.id_evento', 'event', '=');
        $pagination->addFilter('cartoes.id_categoria', 'category', '=');
        $pagination->addFilter('cartoes.situacao', 'status', '=');

        $getLista = (object) $pagination->get();
        $lista = $getLista->data;
        
        $data['recordsTotal'] = $getLista->pagination->rows;
        $data['recordsFiltered'] = $getLista->pagination->rows;
        $data['pagination'] = $getLista->pagination->pages;
        $data['data'] = [];
        foreach ($lista as $value) {
            if ($value->situacao==1) {
                $txSituacao = "<span class=\"badge bg-success\">Cadastro Ativo</span>";
            } else {
                $txSituacao = "<span class=\"badge bg-danger\">Cadastro Bloqueado</span>";
            }
            $txMestre="";
            if ($value->mestre==1) {
                $txMestre = "<span class=\"badge bg-info\">Cartão mestre</span>";
            }
            $createTime = $value->create_time?date('d/m/Y H:i:s', strtotime($value->create_time)):'';
            $updateTime = $value->updated_at?date('d/m/Y H:i:s', strtotime($value->updated_at)):'';
            $linkEdit = base_url() . "/ticket/{$value->id}";
            $src = base_url() . "/ticket/photo/{$value->id}";
            $btg = "<a href=\"{$linkEdit}\" class=\"btn btn-sm rounded-circle btn-primary btn-soft mb-2\" data-id=\"{$value->id}\"  data-toggle=\"tooltip\" data-placement=\"top\" title=\"Editar\" ><i class=\"fi fi-arrow-right\"></i></a> ";
            $groupButton = "<div class=\"btn-group\" role=\"group\" aria-label=\"Opções\">{$btg}</dvi>";
            $evento = "<div class=\"ms-3\">{$value->evento}<small class=\"d-block\">Categoria: {$value->categoria}</small></div>";
            $ticket = "<div class=\"ms-3\">{$value->cartao}<small class=\"d-block\">Nome: {$value->nome}</small></div>";
            $situacao = "<div class=\"ms-3\">{$txSituacao}<small class=\"d-block\">{$txMestre}</small></div>";
            $activity = "<div class=\"ms-3\">Criado em: <span class=\"badge rounded-pill bg-primary\">{$createTime}</span><small class=\"d-block\">Atualizado em: <span class=\"badge rounded-pill bg-secondary\"><strong>{$updateTime}</strong></span></small></div>";
            $image = "<img id=\"photo-{$value->id}\" class=\"img-fluid bg-suprime opacity-9 img-thumbnail\" src=\"{$src}\" />";
            $data['data'][] = [
                $value->id,
                $ticket,
                $evento,
                $situacao,
                $activity,
                $image,
                $groupButton,
            ];
        }
        return $this->response->setJSON($data);
    }

}
